<?php
/**
 * Cleanup Script: Remove invalid soal references
 * 
 * This script finds and removes references to questions that no longer exist
 * in the m_soal table from:
 * - m_soalpaket (paket soal)
 * - m_soalsim (simulasi cepat)
 * 
 * Run this script from browser or CLI to clean up orphaned references.
 */

include '../config.php';

// Check if user is admin
if(!isset($_SESSION['email']) || $_SESSION['role'] != 'admin') {
    die('Access denied. Admin only.');
}

echo "<h2>Cleanup Invalid Soal References</h2>";
echo "<hr>";

// =============================================
// 1. Find and clean invalid references in m_soalpaket
// =============================================
echo "<h3>1. Checking m_soalpaket...</h3>";

$stmt = $conn->query("
    SELECT sp.id, sp.paketsoal_id, sp.soal_id, ps.nama as paket_nama
    FROM m_soalpaket sp
    LEFT JOIN m_soal s ON sp.soal_id = s.id
    LEFT JOIN m_paket_soal ps ON sp.paketsoal_id = ps.id
    WHERE s.id IS NULL
");
$invalid_paket = $stmt->fetchAll(PDO::FETCH_ASSOC);

if(count($invalid_paket) > 0) {
    echo "<p style='color: orange;'>Found <strong>" . count($invalid_paket) . "</strong> invalid references:</p>";
    echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>";
    echo "<tr><th>ID</th><th>Paket ID</th><th>Paket Nama</th><th>Missing Soal ID</th></tr>";
    foreach($invalid_paket as $row) {
        echo "<tr>";
        echo "<td>{$row['id']}</td>";
        echo "<td>{$row['paketsoal_id']}</td>";
        echo "<td>{$row['paket_nama']}</td>";
        echo "<td style='color: red;'>{$row['soal_id']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Delete invalid references
    $delete_ids = array_column($invalid_paket, 'id');
    $placeholders = implode(',', array_fill(0, count($delete_ids), '?'));
    $stmt = $conn->prepare("DELETE FROM m_soalpaket WHERE id IN ($placeholders)");
    $stmt->execute($delete_ids);
    
    echo "<p style='color: green;'>✓ Deleted " . count($invalid_paket) . " invalid references from m_soalpaket.</p>";
} else {
    echo "<p style='color: green;'>✓ No invalid references found in m_soalpaket.</p>";
}

// =============================================
// 2. Find and clean invalid references in m_soalsim
// =============================================
echo "<h3>2. Checking m_soalsim...</h3>";

$stmt = $conn->query("
    SELECT ss.id, ss.simcepat_id, ss.soal_id, sc.nama as sim_nama
    FROM m_soalsim ss
    LEFT JOIN m_soal s ON ss.soal_id = s.id
    LEFT JOIN m_simulasi_cepat sc ON ss.simcepat_id = sc.id
    WHERE s.id IS NULL
");
$invalid_sim = $stmt->fetchAll(PDO::FETCH_ASSOC);

if(count($invalid_sim) > 0) {
    echo "<p style='color: orange;'>Found <strong>" . count($invalid_sim) . "</strong> invalid references:</p>";
    echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>";
    echo "<tr><th>ID</th><th>Simulasi ID</th><th>Simulasi Nama</th><th>Missing Soal ID</th></tr>";
    foreach($invalid_sim as $row) {
        echo "<tr>";
        echo "<td>{$row['id']}</td>";
        echo "<td>{$row['simcepat_id']}</td>";
        echo "<td>{$row['sim_nama']}</td>";
        echo "<td style='color: red;'>{$row['soal_id']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Delete invalid references
    $delete_ids = array_column($invalid_sim, 'id');
    $placeholders = implode(',', array_fill(0, count($delete_ids), '?'));
    $stmt = $conn->prepare("DELETE FROM m_soalsim WHERE id IN ($placeholders)");
    $stmt->execute($delete_ids);
    
    echo "<p style='color: green;'>✓ Deleted " . count($invalid_sim) . " invalid references from m_soalsim.</p>";
} else {
    echo "<p style='color: green;'>✓ No invalid references found in m_soalsim.</p>";
}

// =============================================
// Summary
// =============================================
echo "<hr>";
echo "<h3>Summary</h3>";
echo "<p>Total cleaned from m_soalpaket: <strong>" . count($invalid_paket) . "</strong></p>";
echo "<p>Total cleaned from m_soalsim: <strong>" . count($invalid_sim) . "</strong></p>";
echo "<p style='color: green; font-weight: bold;'>Cleanup completed!</p>";
echo "<p><a href='../view/m_paket_soal/index.php'>← Back to Paket Soal</a></p>";
?>
